<?php

namespace App\Http\Controllers;

use App\Models\CourseReview;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Exception;
use Illuminate\Support\Facades\Log;

class CourseReviewController extends Controller
{
    public function index(): JsonResponse
    {
        try {
            $reviews = CourseReview::with(['user', 'course'])->get();
            return response()->json(['success' => true, 'data' => $reviews, 'message' => 'Reviews retrieved successfully'], 200);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'message' => 'Failed to retrieve reviews: ' . $e->getMessage()], 500);
        }
    }

    public function store(Request $request): JsonResponse
    {
        \Log::info('Request Payload:', $request->all());
        try {
            $validated = $request->validate([
                'User_id' => 'required|exists:users,User_id',
                'Institution' => 'required|string|max:255',
                'Degree' => 'required|string|max:255',
                'Field_of_study' => 'required|string|max:255',
                'Start_date' => 'required|date',
                'End_date' => 'nullable|date|after_or_equal:Start_date',
                'Currently_studying' => 'boolean',
                'Description' => 'nullable|string|max:65535',
            ]);

            $education = EducationBackground::create($validated);
            $education->load('user');
            return response()->json(['success' => true, 'data' => $education, 'message' => 'Education background created successfully'], 201);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json(['success' => false, 'message' => 'Validation failed', 'errors' => $e->errors()], 422);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'message' => 'Failed to create education background: ' . $e->getMessage()], 500);
        }
    }
    public function show($id): JsonResponse
    {
        try {
            $review = CourseReview::with(['user', 'course'])->findOrFail($id);
            return response()->json(['success' => true, 'data' => $review, 'message' => 'Review retrieved successfully'], 200);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'message' => 'Review not found: ' . $e->getMessage()], 404);
        }
    }

    public function update(Request $request, $id): JsonResponse
    {
        try {
            $review = CourseReview::findOrFail($id);
            $validated = $request->validate([
                'User_id' => 'required|exists:users,User_id',
                'Course_id' => 'required|exists:courses,Course_id',
                'Rating' => 'required|integer|between:1,5',
                'Comment' => 'nullable|string|max:65535',
            ]);

            $review->update($validated);
            $review->load(['user', 'course']);
            return response()->json(['success' => true, 'data' => $review, 'message' => 'Review updated successfully'], 200);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'message' => 'Failed to update review: ' . $e->getMessage()], 500);
        }
    }

    public function destroy($id): JsonResponse
    {
        try {
            $review = CourseReview::findOrFail($id);
            $review->delete();
            return response()->json(['success' => true, 'data' => null, 'message' => 'Review deleted successfully'], 204);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'message' => 'Failed to delete review: ' . $e->getMessage()], 500);
        }
    }
}
